-- CRM Sistemi Veritabanı Şeması
-- MySQL 8.0+
-- Tarih: 2024

-- Kullanıcılar Tablosu
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    ad_soyad VARCHAR(255) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    sifre VARCHAR(255) NOT NULL,
    rol ENUM('admin', 'kullanici') NOT NULL DEFAULT 'kullanici',
    durum ENUM('aktif', 'pasif') NOT NULL DEFAULT 'aktif',
    olusturulma_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    guncelleme_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_rol (rol),
    INDEX idx_durum (durum)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Müşteriler Tablosu
CREATE TABLE IF NOT EXISTS customers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    ad_soyad VARCHAR(255) NOT NULL,
    sirket_adi VARCHAR(255),
    telefon VARCHAR(20),
    email VARCHAR(255),
    adres TEXT,
    aciklama TEXT,
    kullanici_id INT,
    durum ENUM('aktif', 'pasif') NOT NULL DEFAULT 'aktif',
    olusturulma_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    guncelleme_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (kullanici_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_kullanici (kullanici_id),
    INDEX idx_durum (durum),
    INDEX idx_ad_soyad (ad_soyad),
    INDEX idx_sirket (sirket_adi)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Görevler Tablosu
CREATE TABLE IF NOT EXISTS tasks (
    id INT PRIMARY KEY AUTO_INCREMENT,
    gorev VARCHAR(255) NOT NULL,
    aciklama TEXT,
    son_tarih DATE,
    durum ENUM('bekliyor', 'yapiliyor', 'tamamlandi') NOT NULL DEFAULT 'bekliyor',
    oncelik ENUM('dusuk', 'orta', 'yuksek') NOT NULL DEFAULT 'orta',
    atayan_admin_id INT,
    atanmis_kullanici_id INT,
    musteri_id INT,
    olusturulma_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    guncelleme_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (atayan_admin_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (atanmis_kullanici_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (musteri_id) REFERENCES customers(id) ON DELETE CASCADE,
    INDEX idx_atanan (atanmis_kullanici_id),
    INDEX idx_durum (durum),
    INDEX idx_son_tarih (son_tarih),
    INDEX idx_oncelik (oncelik)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Müşteri Notları Tablosu
CREATE TABLE IF NOT EXISTS customer_notes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    customer_id INT NOT NULL,
    user_id INT NOT NULL,
    not TEXT NOT NULL,
    tarih TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_customer (customer_id),
    INDEX idx_user (user_id),
    INDEX idx_tarih (tarih)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Log Kayıtları Tablosu
CREATE TABLE IF NOT EXISTS logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    islem VARCHAR(255) NOT NULL,
    detay TEXT,
    ip_adresi VARCHAR(45),
    tarih TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_tarih (tarih),
    INDEX idx_islem (islem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Bildirimler Tablosu
CREATE TABLE IF NOT EXISTS notifications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    baslik VARCHAR(255) NOT NULL,
    mesaj TEXT NOT NULL,
    okundu BOOLEAN DEFAULT FALSE,
    olusturulma_tarihi TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_okundu (user_id, okundu),
    INDEX idx_tarih (olusturulma_tarihi)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Demo Kullanıcılar Ekle
-- Şifreler: admin123 ve user123
INSERT INTO users (ad_soyad, email, sifre, rol, durum) VALUES
('Sistem Admin', 'admin@crm.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'aktif'),
('Ahmet Yılmaz', 'ahmet@crm.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'kullanici', 'aktif')
ON DUPLICATE KEY UPDATE id=id;

-- Demo Müşteriler Ekle
INSERT INTO customers (ad_soyad, sirket_adi, telefon, email, adres, kullanici_id, aciklama) VALUES
('Mehmet Demir', 'Demir Yazılım A.Ş.', '0555 123 4567', 'mehmet@demiryazilim.com', 'İstanbul, Türkiye', 2, 'Yazılım geliştirme şirketi'),
('Ayşe Kaya', 'Kaya Teknoloji', '0533 987 6543', 'ayse@kayatek.com', 'Ankara, Türkiye', 2, 'IT danışmanlık'),
('Fatma Şahin', 'Şahin Danışmanlık', '0542 555 1234', 'fatma@sahindanismanlik.com', 'İzmir, Türkiye', 2, 'İş danışmanlığı')
ON DUPLICATE KEY UPDATE id=id;

-- Demo Görevler Ekle
INSERT INTO tasks (gorev, aciklama, son_tarih, durum, oncelik, atayan_admin_id, atanmis_kullanici_id, musteri_id) VALUES
('Müşteri ile görüşme', 'Yeni proje için müşteri ile telefon görüşmesi yapılacak', '2025-01-20', 'bekliyor', 'yuksek', 1, 2, 1),
('Teklif hazırlama', 'Yazılım projesi için detaylı teklif hazırlanacak', '2025-01-25', 'yapiliyor', 'orta', 1, 2, 2),
('Demo sunumu', 'Ürün demosu yapılacak', '2025-02-01', 'bekliyor', 'dusuk', 1, 2, 3)
ON DUPLICATE KEY UPDATE id=id;

-- Demo Notlar Ekle
INSERT INTO customer_notes (customer_id, user_id, not) VALUES
(1, 2, 'İlk görüşme gerçekleştirildi. Müşteri proje ile ilgileniyor.'),
(2, 2, 'Teklif gönderildi. Geri dönüş bekleniyor.'),
(3, 2, 'Demo tarihi belirlendi: 01.02.2025')
ON DUPLICATE KEY UPDATE id=id;

-- Optimizasyon için ek indeksler
ALTER TABLE users ADD INDEX idx_created (olusturulma_tarihi);
ALTER TABLE customers ADD INDEX idx_created (olusturulma_tarihi);
ALTER TABLE tasks ADD INDEX idx_created (olusturulma_tarihi);

-- Veritabanı karakter seti kontrolü
ALTER DATABASE CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;