<?php
/**
 * CRM Sistemi - Süper Detaylı Kurulum Kontrol
 * Tüm dosyaları, konfigürasyonları ve bağlantıları kontrol eder
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

$checks = [];
$errors = [];
$warnings = [];

// ============================================
// KONTROL 1: PHP VERSİYONU
// ============================================
$phpVersion = phpversion();
$checks[] = [
    'title' => 'PHP Versiyonu',
    'status' => version_compare($phpVersion, '8.0.0', '>=') ? 'success' : 'error',
    'message' => "PHP $phpVersion " . (version_compare($phpVersion, '8.0.0', '>=') ? '✓' : '✗ (Minimum PHP 8.0 gerekli)'),
    'detail' => "Yüklü versiyon: $phpVersion\nGerekli: 8.0+"
];

if (!version_compare($phpVersion, '8.0.0', '>=')) {
    $errors[] = "PHP versiyonu 8.0'dan düşük!";
}

// ============================================
// KONTROL 2: GEREKLİ PHP EXTENSİONLARI
// ============================================
$requiredExtensions = ['pdo', 'pdo_mysql', 'json', 'mbstring', 'openssl'];
foreach ($requiredExtensions as $ext) {
    $loaded = extension_loaded($ext);
    $checks[] = [
        'title' => "PHP Extension: $ext",
        'status' => $loaded ? 'success' : 'error',
        'message' => $loaded ? "$ext yüklü ✓" : "$ext yüklü değil ✗",
        'detail' => ''
    ];
    if (!$loaded) {
        $errors[] = "PHP extension eksik: $ext";
    }
}

// ============================================
// KONTROL 3: DOSYA YAPISI
// ============================================
$requiredFiles = [
    'config.php' => 'Ana yapılandırma dosyası',
    '.htaccess' => 'Apache yapılandırması',
    'public/index.php' => 'API giriş noktası',
    'public/login.html' => 'Giriş sayfası',
    'routes/api.php' => 'API yönlendirmeleri',
    'app/Core/Database.php' => 'Veritabanı sınıfı',
    'app/Core/Router.php' => 'Router sınıfı',
    'app/Core/JWT.php' => 'JWT sınıfı',
    'app/Core/Request.php' => 'Request sınıfı',
    'app/Core/Response.php' => 'Response sınıfı',
    'app/Models/User.php' => 'User modeli',
    'app/Models/Customer.php' => 'Customer modeli',
    'app/Models/Task.php' => 'Task modeli',
    'app/Models/Log.php' => 'Log modeli',
    'app/Models/Note.php' => 'Note modeli',
    'app/Controllers/AuthController.php' => 'Auth controller',
    'app/Controllers/AdminController.php' => 'Admin controller',
    'app/Controllers/CustomerController.php' => 'Customer controller',
    'app/Controllers/TaskController.php' => 'Task controller',
    'app/Controllers/NoteController.php' => 'Note controller',
    'app/Middleware/AuthMiddleware.php' => 'Auth middleware',
    'app/Middleware/AdminMiddleware.php' => 'Admin middleware',
    'config/app.php' => 'App config',
    'config/database.php' => 'Database config',
    'public/assets/css/admin.css' => 'Admin CSS',
    'public/assets/css/user.css' => 'User CSS',
    'public/assets/js/common.js' => 'Common JS'
];

foreach ($requiredFiles as $file => $description) {
    $fullPath = __DIR__ . '/' . $file;
    $exists = file_exists($fullPath);
    $readable = $exists ? is_readable($fullPath) : false;
    
    $status = 'error';
    $message = '';
    $detail = "Dosya yolu: $fullPath\n";
    
    if ($exists && $readable) {
        $status = 'success';
        $size = filesize($fullPath);
        $message = "✓ $description ($size bytes)";
        $detail .= "Boyut: $size bytes\n";
        $detail .= "İzinler: " . substr(sprintf('%o', fileperms($fullPath)), -4);
        
        if (in_array($file, ['config.php', 'public/index.php', 'app/Models/Log.php', 'app/Models/Note.php'])) {
            $content = file_get_contents($fullPath);
            $detail .= "\n\nİlk 200 karakter:\n" . substr($content, 0, 200);
            
            if (strpos($file, 'app/') === 0 && pathinfo($file, PATHINFO_EXTENSION) === 'php') {
                if (strpos($content, '<?php') === false) {
                    $status = 'error';
                    $message = "✗ $description - PHP opening tag yok!";
                    $errors[] = "$file dosyasında <?php tag eksik!";
                }
            }
        }
    } elseif ($exists && !$readable) {
        $status = 'warning';
        $message = "⚠ $description - Dosya okunamıyor";
        $detail .= "İzin sorunu!";
        $warnings[] = "$file dosyası okunamıyor";
    } else {
        $message = "✗ $description - Dosya bulunamadı!";
        $detail .= "DOSYA YOK!";
        $errors[] = "$file dosyası eksik!";
    }
    
    $checks[] = [
        'title' => $file,
        'status' => $status,
        'message' => $message,
        'detail' => $detail
    ];
}

// VERİTABANI KONTROLÜ
if (file_exists(__DIR__ . '/config.php')) {
    try {
        $config = require __DIR__ . '/config.php';
        $dbConfig = $config['database'];
        
        $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset=utf8mb4";
        $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        $checks[] = [
            'title' => 'Veritabanı Bağlantısı',
            'status' => 'success',
            'message' => "✓ Bağlantı başarılı",
            'detail' => "DB: {$dbConfig['database']}"
        ];
        
        $requiredTables = ['users', 'customers', 'tasks', 'customer_notes', 'logs', 'notifications'];
        $stmt = $pdo->query("SHOW TABLES");
        $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($requiredTables as $table) {
            if (in_array($table, $existingTables)) {
                $countStmt = $pdo->query("SELECT COUNT(*) as count FROM `$table`");
                $count = $countStmt->fetch(PDO::FETCH_ASSOC)['count'];
                
                $checks[] = [
                    'title' => "Tablo: $table",
                    'status' => 'success',
                    'message' => "✓ Mevcut ($count kayıt)",
                    'detail' => "Kayıt: $count"
                ];
            } else {
                $checks[] = [
                    'title' => "Tablo: $table",
                    'status' => 'error',
                    'message' => "✗ Tablo yok",
                    'detail' => "SQL çalıştırın!"
                ];
                $errors[] = "$table tablosu eksik!";
            }
        }
        
    } catch (PDOException $e) {
        $checks[] = [
            'title' => 'Veritabanı',
            'status' => 'error',
            'message' => "✗ Hata: " . $e->getMessage(),
            'detail' => $e->getMessage()
        ];
        $errors[] = "DB hatası: " . $e->getMessage();
    }
}

$totalChecks = count($checks);
$successCount = count(array_filter($checks, fn($c) => $c['status'] === 'success'));
$errorCount = count(array_filter($checks, fn($c) => $c['status'] === 'error'));
$warningCount = count(array_filter($checks, fn($c) => $c['status'] === 'warning'));
$overallStatus = $errorCount === 0 ? 'success' : 'error';
$percentage = round(($successCount / $totalChecks) * 100);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <title>CRM Kurulum Kontrol</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 20px; }
        .container { max-width: 1200px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 20px 60px rgba(0,0,0,0.3); }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 40px; text-align: center; }
        .header h1 { font-size: 32px; margin-bottom: 10px; }
        .progress-bar { background: rgba(255,255,255,0.2); height: 30px; border-radius: 15px; margin-top: 20px; overflow: hidden; }
        .progress-fill { background: <?php echo $overallStatus === 'success' ? '#10b981' : '#ef4444'; ?>; height: 100%; width: <?php echo $percentage; ?>%; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; }
        .stats { display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; padding: 30px; background: #f8fafc; }
        .stat-card { background: white; padding: 20px; border-radius: 12px; text-align: center; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .stat-card h3 { font-size: 14px; color: #6b7280; margin-bottom: 10px; }
        .stat-card .value { font-size: 32px; font-weight: bold; }
        .stat-card.success .value { color: #10b981; }
        .stat-card.error .value { color: #ef4444; }
        .stat-card.warning .value { color: #f59e0b; }
        .checks-container { padding: 30px; max-height: 600px; overflow-y: auto; }
        .check-item { background: #f9fafb; padding: 15px; margin-bottom: 10px; border-radius: 8px; border-left: 4px solid #ccc; }
        .check-item.success { border-left-color: #10b981; background: #f0fdf4; }
        .check-item.error { border-left-color: #ef4444; background: #fef2f2; }
        .check-item.warning { border-left-color: #f59e0b; background: #fffbeb; }
        .check-title { font-weight: 600; color: #1f2937; display: block; margin-bottom: 5px; }
        .check-message { color: #4b5563; font-size: 14px; }
        .summary { padding: 30px; background: <?php echo $overallStatus === 'success' ? '#f0fdf4' : '#fef2f2'; ?>; text-align: center; }
        .summary h2 { font-size: 28px; margin-bottom: 20px; color: <?php echo $overallStatus === 'success' ? '#10b981' : '#ef4444'; ?>; }
        .btn { display: inline-block; padding: 12px 30px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-decoration: none; border-radius: 8px; margin: 10px; font-weight: 600; }
        .error-list { text-align: left; max-width: 800px; margin: 20px auto; padding: 20px; background: white; border-radius: 8px; }
        .error-list li { margin: 10px 0; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔍 CRM Kurulum Kontrol</h1>
            <p>Detaylı sistem kontrolü yapılıyor...</p>
            <div class="progress-bar">
                <div class="progress-fill"><?php echo $percentage; ?>%</div>
            </div>
        </div>

        <div class="stats">
            <div class="stat-card">
                <h3>Toplam</h3>
                <div class="value"><?php echo $totalChecks; ?></div>
            </div>
            <div class="stat-card success">
                <h3>Başarılı</h3>
                <div class="value"><?php echo $successCount; ?></div>
            </div>
            <div class="stat-card error">
                <h3>Hata</h3>
                <div class="value"><?php echo $errorCount; ?></div>
            </div>
            <div class="stat-card warning">
                <h3>Uyarı</h3>
                <div class="value"><?php echo $warningCount; ?></div>
            </div>
        </div>

        <div class="checks-container">
            <?php foreach ($checks as $check): ?>
                <div class="check-item <?php echo $check['status']; ?>">
                    <span class="check-title"><?php echo htmlspecialchars($check['title']); ?></span>
                    <div class="check-message"><?php echo htmlspecialchars($check['message']); ?></div>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="summary">
            <?php if ($overallStatus === 'success'): ?>
                <h2>🎉 Kurulum Başarılı!</h2>
                <p>Sistem kullanıma hazır!</p>
                <a href="/login.html" class="btn">Giriş Sayfası →</a>
                <p style="margin-top: 20px; color: #6b7280;">
                    Admin: admin@crm.com / admin123<br>
                    User: ahmet@crm.com / user123
                </p>
            <?php else: ?>
                <h2>❌ Hatalar Var</h2>
                <p>Lütfen düzeltin:</p>
                <?php if (!empty($errors)): ?>
                    <div class="error-list">
                        <h3>Hatalar:</h3>
                        <ul>
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>