<?php
// app/Models/Log.php

namespace App\Models;

use App\Core\Database;

class Log {
    private $db;
    private $table = 'logs';

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Log kaydı oluştur
     * @param int|null $userId Kullanıcı ID
     * @param string $action İşlem adı
     * @param string $description Açıklama
     * @param string|null $ipAddress IP adresi
     */
    public function create($userId, $action, $description = '', $ipAddress = null) {
        $data = [
            'user_id' => $userId,
            'action' => $action,
            'description' => $description,
            'ip_address' => $ipAddress ?? ($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert($this->table, $data);
    }

    /**
     * Tüm logları getir
     * @param int $limit Limit
     */
    public function findAll($limit = 100) {
        $sql = "SELECT l.*, u.ad_soyad as kullanici_adi 
                FROM {$this->table} l 
                LEFT JOIN users u ON l.user_id = u.id 
                ORDER BY l.created_at DESC 
                LIMIT :limit";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Kullanıcıya göre logları getir
     */
    public function findByUser($userId, $limit = 50) {
        $sql = "SELECT * FROM {$this->table} 
                WHERE user_id = :user_id 
                ORDER BY created_at DESC 
                LIMIT :limit";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bindValue(':user_id', $userId, \PDO::PARAM_INT);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * İşlem tipine göre logları getir
     */
    public function findByAction($action, $limit = 50) {
        $sql = "SELECT l.*, u.ad_soyad as kullanici_adi 
                FROM {$this->table} l 
                LEFT JOIN users u ON l.user_id = u.id 
                WHERE l.action = :action 
                ORDER BY l.created_at DESC 
                LIMIT :limit";
        
        $stmt = $this->db->getConnection()->prepare($sql);
        $stmt->bindValue(':action', $action);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Tarih aralığına göre logları getir
     */
    public function findByDateRange($startDate, $endDate) {
        $sql = "SELECT l.*, u.ad_soyad as kullanici_adi 
                FROM {$this->table} l 
                LEFT JOIN users u ON l.user_id = u.id 
                WHERE l.created_at BETWEEN :start_date AND :end_date 
                ORDER BY l.created_at DESC";
        
        return $this->db->fetchAll($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
    }

    /**
     * Eski logları temizle
     */
    public function cleanup($daysOld = 30) {
        $date = date('Y-m-d H:i:s', strtotime("-{$daysOld} days"));
        return $this->db->delete($this->table, 'created_at < :date', ['date' => $date]);
    }
}