<?php
// app/Core/Database.php - Tam Versiyon

namespace App\Core;

use PDO;
use PDOException;

class Database {
    private static $instance = null;
    private $connection;

    private function __construct() {
        // config.php dosyasını yükle
        $config = require __DIR__ . '/../../config.php';
        $dbConfig = $config['database'];
        
        try {
            $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset={$dbConfig['charset']}";
            $this->connection = new PDO($dsn, $dbConfig['username'], $dbConfig['password'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);
        } catch (PDOException $e) {
            // Hatayı logla
            self::logError($e, $dbConfig);
            
            // JSON API yanıtı
            header('Content-Type: application/json; charset=utf-8');
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Veritabanı bağlantı hatası',
                'error' => $config['app']['debug'] ? $e->getMessage() : null
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }

    /**
     * Bağlantı hatalarını logla
     */
    private static function logError(PDOException $e, array $context = []) {
        $logDir = __DIR__ . '/../../storage/logs';
        $logFile = $logDir . '/database_errors.log';

        // Log klasörünü oluştur
        if (!is_dir($logDir)) {
            @mkdir($logDir, 0755, true);
        }

        $timestamp = date('Y-m-d H:i:s');
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'CLI';
        $uri = $_SERVER['REQUEST_URI'] ?? 'N/A';

        $logMessage = "[{$timestamp}] DATABASE_CONNECTION_ERROR\n";
        $logMessage .= "IP: {$ip}\n";
        $logMessage .= "URI: {$uri}\n";
        $logMessage .= "Error: {$e->getMessage()}\n";
        $logMessage .= "Host: " . ($context['host'] ?? 'N/A') . "\n";
        $logMessage .= "Database: " . ($context['database'] ?? 'N/A') . "\n";
        $logMessage .= str_repeat("-", 60) . "\n\n";

        @file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->connection;
    }

    /**
     * SQL sorgusu çalıştır
     */
    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("SQL Hatası: " . $e->getMessage() . " | SQL: " . $sql);
            throw $e;
        }
    }

    /**
     * Tüm sonuçları getir
     */
    public function fetchAll($sql, $params = []) {
        return $this->query($sql, $params)->fetchAll();
    }

    /**
     * Tek satır getir
     */
    public function fetch($sql, $params = []) {
        return $this->query($sql, $params)->fetch();
    }

    /**
     * Tabloya veri ekle
     */
    public function insert($table, $data) {
        $keys = array_keys($data);
        $fields = implode(', ', $keys);
        $placeholders = ':' . implode(', :', $keys);
        
        $sql = "INSERT INTO {$table} ({$fields}) VALUES ({$placeholders})";
        $this->query($sql, $data);
        return $this->connection->lastInsertId();
    }

    /**
     * Veri güncelle
     */
    public function update($table, $data, $where, $whereParams = []) {
        $set = [];
        foreach (array_keys($data) as $key) {
            $set[] = "{$key} = :{$key}";
        }
        $setString = implode(', ', $set);
        
        $sql = "UPDATE {$table} SET {$setString} WHERE {$where}";
        $params = array_merge($data, $whereParams);
        return $this->query($sql, $params)->rowCount();
    }

    /**
     * Veri sil
     */
    public function delete($table, $where, $params = []) {
        $sql = "DELETE FROM {$table} WHERE {$where}";
        return $this->query($sql, $params)->rowCount();
    }

    /**
     * Kayıt sayısı
     */
    public function count($table, $where = '1=1', $params = []) {
        $sql = "SELECT COUNT(*) as total FROM {$table} WHERE {$where}";
        $result = $this->fetch($sql, $params);
        return $result ? (int)$result['total'] : 0;
    }
}