<?php
// app/Controllers/NoteController.php

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Models\Note;
use App\Models\Customer;
use App\Models\Log;
use App\Middleware\AuthMiddleware;

class NoteController {
    private $noteModel;
    private $customerModel;
    private $logModel;

    public function __construct() {
        $this->noteModel = new Note();
        $this->customerModel = new Customer();
        $this->logModel = new Log();
    }

    public function index($customerId) {
        AuthMiddleware::handle();

        // Müşteri kontrolü
        $customer = $this->customerModel->findById($customerId);
        if (!$customer) {
            Response::notFound('Müşteri bulunamadı');
        }

        // Yetki kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $customer['kullanici_id'] != $userId) {
            Response::forbidden('Bu müşterinin notlarına erişim yetkiniz yok');
        }

        $notes = $this->noteModel->findByCustomer($customerId);
        Response::success('Notlar listelendi', $notes);
    }

    public function create($customerId) {
        AuthMiddleware::handle();

        // Müşteri kontrolü
        $customer = $this->customerModel->findById($customerId);
        if (!$customer) {
            Response::notFound('Müşteri bulunamadı');
        }

        // Yetki kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $customer['kullanici_id'] != $userId) {
            Response::forbidden('Bu müşteriye not ekleme yetkiniz yok');
        }

        $request = new Request();
        
        $validation = $request->validate([
            'not' => 'required|min:5'
        ]);

        if ($validation !== true) {
            Response::error('Doğrulama hatası', $validation);
        }

        $data = [
            'customer_id' => $customerId,
            'user_id' => $userId,
            'not' => $request->input('not')
        ];

        $noteId = $this->noteModel->create($data);
        $this->logModel->create($userId, 'Not ekleme', "Müşteri ID: $customerId için not eklendi");

        Response::success('Not başarıyla eklendi', ['id' => $noteId], 201);
    }

    public function delete($id) {
        AuthMiddleware::handle();

        $this->noteModel->delete($id);
        
        $userId = AuthMiddleware::id();
        $this->logModel->create($userId, 'Not silme', "Not ID: $id silindi");

        Response::success('Not başarıyla silindi');
    }
}