<?php
// app/Controllers/CustomerController.php

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Models\Customer;
use App\Models\Log;
use App\Middleware\AuthMiddleware;

class CustomerController {
    private $customerModel;
    private $logModel;

    public function __construct() {
        $this->customerModel = new Customer();
        $this->logModel = new Log();
    }

    public function index() {
        AuthMiddleware::handle();

        $userId = AuthMiddleware::id();
        $role = AuthMiddleware::role();

        if ($role === 'admin') {
            $customers = $this->customerModel->findAll();
        } else {
            $customers = $this->customerModel->findByUser($userId);
        }

        Response::success('Müşteriler listelendi', $customers);
    }

    public function show($id) {
        AuthMiddleware::handle();

        $customer = $this->customerModel->findById($id);
        
        if (!$customer) {
            Response::notFound('Müşteri bulunamadı');
        }

        // Yetkisiz erişim kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $customer['kullanici_id'] != $userId) {
            Response::forbidden('Bu müşteriye erişim yetkiniz yok');
        }

        Response::success('Müşteri detayları', $customer);
    }

    public function create() {
        AuthMiddleware::handle();

        $request = new Request();
        
        $validation = $request->validate([
            'ad_soyad' => 'required|min:3',
            'telefon' => 'required'
        ]);

        if ($validation !== true) {
            Response::error('Doğrulama hatası', $validation);
        }

        $data = [
            'ad_soyad' => $request->input('ad_soyad'),
            'sirket_adi' => $request->input('sirket_adi'),
            'telefon' => $request->input('telefon'),
            'email' => $request->input('email'),
            'adres' => $request->input('adres'),
            'aciklama' => $request->input('aciklama'),
            'kullanici_id' => $request->input('kullanici_id')
        ];

        // Normal kullanıcı sadece kendi adına müşteri ekleyebilir
        $role = AuthMiddleware::role();
        if ($role !== 'admin') {
            $data['kullanici_id'] = AuthMiddleware::id();
        }

        $customerId = $this->customerModel->create($data);
        
        $userId = AuthMiddleware::id();
        $this->logModel->create($userId, 'Müşteri oluşturma', "Yeni müşteri: {$data['ad_soyad']}");

        Response::success('Müşteri başarıyla oluşturuldu', ['id' => $customerId], 201);
    }

    public function update($id) {
        AuthMiddleware::handle();

        $customer = $this->customerModel->findById($id);
        
        if (!$customer) {
            Response::notFound('Müşteri bulunamadı');
        }

        // Yetkisiz erişim kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $customer['kullanici_id'] != $userId) {
            Response::forbidden('Bu müşteriyi düzenleme yetkiniz yok');
        }

        $request = new Request();
        
        $data = [];
        
        if ($request->has('ad_soyad')) $data['ad_soyad'] = $request->input('ad_soyad');
        if ($request->has('sirket_adi')) $data['sirket_adi'] = $request->input('sirket_adi');
        if ($request->has('telefon')) $data['telefon'] = $request->input('telefon');
        if ($request->has('email')) $data['email'] = $request->input('email');
        if ($request->has('adres')) $data['adres'] = $request->input('adres');
        if ($request->has('aciklama')) $data['aciklama'] = $request->input('aciklama');
        
        // Admin kullanıcı atayabilir
        if ($role === 'admin' && $request->has('kullanici_id')) {
            $data['kullanici_id'] = $request->input('kullanici_id');
        }

        if (empty($data)) {
            Response::error('Güncellenecek veri bulunamadı');
        }

        $this->customerModel->update($id, $data);
        $this->logModel->create($userId, 'Müşteri güncelleme', "Müşteri ID: $id güncellendi");

        Response::success('Müşteri başarıyla güncellendi');
    }

    public function delete($id) {
        AuthMiddleware::handle();

        $customer = $this->customerModel->findById($id);
        
        if (!$customer) {
            Response::notFound('Müşteri bulunamadı');
        }

        // Yetkisiz erişim kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $customer['kullanici_id'] != $userId) {
            Response::forbidden('Bu müşteriyi silme yetkiniz yok');
        }

        $this->customerModel->delete($id);
        $this->logModel->create($userId, 'Müşteri silme', "Müşteri ID: $id silindi");

        Response::success('Müşteri başarıyla silindi');
    }

    public function search() {
        AuthMiddleware::handle();

        $request = new Request();
        $term = $request->input('term');

        if (!$term) {
            Response::error('Arama terimi giriniz');
        }

        $role = AuthMiddleware::role();
        $userId = $role === 'admin' ? null : AuthMiddleware::id();

        $customers = $this->customerModel->search($term, $userId);

        Response::success('Arama sonuçları', $customers);
    }
}