<?php
// app/Controllers/AuthController.php

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Core\JWT;
use App\Models\User;
use App\Models\Log;
use App\Middleware\AuthMiddleware;

class AuthController {
    private $userModel;
    private $logModel;

    public function __construct() {
        $this->userModel = new User();
        $this->logModel = new Log();
    }

    public function login() {
        $request = new Request();
        
        $validation = $request->validate([
            'email' => 'required|email',
            'sifre' => 'required'
        ]);

        if ($validation !== true) {
            Response::error('Doğrulama hatası', $validation);
        }

        $email = $request->input('email');
        $sifre = $request->input('sifre');

        $user = $this->userModel->findByEmail($email);

        if (!$user) {
            $this->logModel->create(null, 'Başarısız giriş denemesi', "E-posta: $email", $request->ip());
            Response::error('E-posta veya şifre hatalı', null, 401);
        }

        if ($user['durum'] !== 'aktif') {
            Response::error('Hesabınız pasif durumda', null, 403);
        }

        if (!$this->userModel->verifyPassword($sifre, $user['sifre'])) {
            $this->logModel->create($user['id'], 'Başarısız giriş denemesi', "Hatalı şifre", $request->ip());
            Response::error('E-posta veya şifre hatalı', null, 401);
        }

        // JWT token oluştur
        $payload = [
            'id' => $user['id'],
            'email' => $user['email'],
            'rol' => $user['rol'],
            'ad_soyad' => $user['ad_soyad']
        ];

        $token = JWT::encode($payload);

        // Başarılı girişi logla
        $this->logModel->create($user['id'], 'Başarılı giriş', 'Kullanıcı sisteme giriş yaptı', $request->ip());

        Response::success('Giriş başarılı', [
            'token' => $token,
            'user' => [
                'id' => $user['id'],
                'ad_soyad' => $user['ad_soyad'],
                'email' => $user['email'],
                'rol' => $user['rol']
            ]
        ]);
    }

    public function me() {
        AuthMiddleware::handle();
        
        $userId = AuthMiddleware::id();
        $user = $this->userModel->findById($userId);

        if (!$user) {
            Response::notFound('Kullanıcı bulunamadı');
        }

        Response::success('Kullanıcı bilgileri', $user);
    }

    public function updateProfile() {
        AuthMiddleware::handle();
        
        $request = new Request();
        $userId = AuthMiddleware::id();

        $data = [];
        
        if ($request->has('ad_soyad')) {
            $data['ad_soyad'] = $request->input('ad_soyad');
        }

        if ($request->has('email')) {
            $data['email'] = $request->input('email');
        }

        if ($request->has('sifre')) {
            $data['sifre'] = $request->input('sifre');
        }

        if (empty($data)) {
            Response::error('Güncellenecek veri bulunamadı');
        }

        $this->userModel->update($userId, $data);
        $this->logModel->create($userId, 'Profil güncelleme', 'Kullanıcı profilini güncelledi');

        Response::success('Profil başarıyla güncellendi');
    }

    public function logout() {
        AuthMiddleware::handle();
        
        $userId = AuthMiddleware::id();
        $this->logModel->create($userId, 'Çıkış', 'Kullanıcı sistemden çıkış yaptı');

        Response::success('Çıkış başarılı');
    }
}