<?php
// app/Controllers/AdminController.php

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Models\User;
use App\Models\Customer;
use App\Models\Task;
use App\Models\Log;
use App\Middleware\AdminMiddleware;
use App\Middleware\AuthMiddleware;

class AdminController {
    private $userModel;
    private $customerModel;
    private $taskModel;
    private $logModel;

    public function __construct() {
        $this->userModel = new User();
        $this->customerModel = new Customer();
        $this->taskModel = new Task();
        $this->logModel = new Log();
    }

    public function dashboard() {
        AdminMiddleware::handle();

        $stats = [
            'toplam_kullanici' => $this->userModel->count(),
            'toplam_musteri' => $this->customerModel->count(),
            'toplam_gorev' => $this->taskModel->count(),
            'bekleyen_gorev' => $this->taskModel->countByStatus('bekliyor'),
            'yapilan_gorev' => $this->taskModel->countByStatus('yapiliyor'),
            'tamamlanan_gorev' => $this->taskModel->countByStatus('tamamlandi')
        ];

        Response::success('Dashboard verileri', $stats);
    }

    // Kullanıcı Yönetimi
    public function getUsers() {
        AdminMiddleware::handle();

        $users = $this->userModel->findAll();
        Response::success('Kullanıcılar listelendi', $users);
    }

    public function createUser() {
        AdminMiddleware::handle();
        
        $request = new Request();
        
        $validation = $request->validate([
            'ad_soyad' => 'required|min:3',
            'email' => 'required|email',
            'sifre' => 'required|min:6',
            'rol' => 'required'
        ]);

        if ($validation !== true) {
            Response::error('Doğrulama hatası', $validation);
        }

        // E-posta kontrolü
        $existingUser = $this->userModel->findByEmail($request->input('email'));
        if ($existingUser) {
            Response::error('Bu e-posta adresi zaten kullanılıyor');
        }

        $data = [
            'ad_soyad' => $request->input('ad_soyad'),
            'email' => $request->input('email'),
            'sifre' => $request->input('sifre'),
            'rol' => $request->input('rol'),
            'durum' => $request->input('durum', 'aktif')
        ];

        $userId = $this->userModel->create($data);
        
        $adminId = AuthMiddleware::id();
        $this->logModel->create($adminId, 'Kullanıcı oluşturma', "Yeni kullanıcı: {$data['email']}");

        Response::success('Kullanıcı başarıyla oluşturuldu', ['id' => $userId], 201);
    }

    public function updateUser($id) {
        AdminMiddleware::handle();
        
        $request = new Request();

        $user = $this->userModel->findById($id);
        if (!$user) {
            Response::notFound('Kullanıcı bulunamadı');
        }

        $data = [];
        
        if ($request->has('ad_soyad')) {
            $data['ad_soyad'] = $request->input('ad_soyad');
        }
        if ($request->has('email')) {
            $data['email'] = $request->input('email');
        }
        if ($request->has('sifre')) {
            $data['sifre'] = $request->input('sifre');
        }
        if ($request->has('rol')) {
            $data['rol'] = $request->input('rol');
        }
        if ($request->has('durum')) {
            $data['durum'] = $request->input('durum');
        }

        if (empty($data)) {
            Response::error('Güncellenecek veri bulunamadı');
        }

        $this->userModel->update($id, $data);
        
        $adminId = AuthMiddleware::id();
        $this->logModel->create($adminId, 'Kullanıcı güncelleme', "Kullanıcı ID: $id güncellendi");

        Response::success('Kullanıcı başarıyla güncellendi');
    }

    public function deleteUser($id) {
        AdminMiddleware::handle();
        
        $user = $this->userModel->findById($id);
        if (!$user) {
            Response::notFound('Kullanıcı bulunamadı');
        }

        // Kendi hesabını silemez
        if ($id == AuthMiddleware::id()) {
            Response::error('Kendi hesabınızı silemezsiniz');
        }

        $this->userModel->delete($id);
        
        $adminId = AuthMiddleware::id();
        $this->logModel->create($adminId, 'Kullanıcı silme', "Kullanıcı ID: $id silindi");

        Response::success('Kullanıcı başarıyla silindi');
    }

    // Log Görüntüleme
    public function getLogs() {
        AdminMiddleware::handle();

        $logs = $this->logModel->findAll(100);
        Response::success('Loglar listelendi', $logs);
    }
}